/* Xmsg.c - (C) thomas@apestaart.org
 * 
 * version 0.4
 *
 * last changed : 13/05/2001
 *
 * displays a graphical message which disappears over time
 * also displays an .xpm icon in a 50x50 window
 * 
 */

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* project includes */

#include "config.h"

/* default values */

#define IMG_X 0
#define IMG_Y 0
#define IMG_W 50
#define IMG_H 50

#define MSG_X 50
#define MSG_Y 0
#define MSG_W 950
#define MSG_H 50

/* debug ? */
char debug = 0;
#define DEBUG(x) if (debug) { printf ("DEBUG: "); printf (x); }


/* create two windows, one for the icon and one for the message
 * put them next to each other
 * display the icon in the first
 * display the message graphically in the second
 */

/* function prototypes */

Window CreateWindow (Display* dpy, int screen_num, int x, int y, int width, int height);
int LoadFont (Display* display, cfg_t* cfg);
void usage (const char *text);


Window 
CreateWindow (Display* dpy, int screen_num, 
              int x, int y, int width, int height)
{
    /* Create a window and map it */
    
    int vmask;
    XSetWindowAttributes attribs;
    Window w;

    /* message window */
    vmask = CWSaveUnder|CWOverrideRedirect|CWColormap|CWBorderPixel;
    attribs.save_under = True;
    attribs.override_redirect = True;
    attribs.colormap = DefaultColormap (dpy, screen_num);
    attribs.background_pixel = 1000; 
    attribs.border_pixel = 0; /* do not care */

    w = XCreateWindow (dpy, DefaultRootWindow (dpy), x, y, width, height,
                       0, CopyFromParent, CopyFromParent, CopyFromParent, 
                       vmask, &attribs);

    XMapWindow (dpy, w);
    return w;
}

int 
LoadFont (Display* display, cfg_t* cfg)
{
  /* try to load the given font. */
  DEBUG ("loading font\n");

  cfg->font_info = XLoadQueryFont(display, cfg->font_name);
  if (!cfg->font_info) 
  {
      fprintf(stderr, "XLoadQueryFont: failed loading font '%s'\n", cfg->font_name);
      exit(-1);
  }
  /* find the height of the characters drawn using this font.        */
  cfg->font_height = cfg->font_info->ascent + cfg->font_info->descent;
  return 0;
}

GC
CreateGC (Display* display, Window win, char color[])
{
  /* Create a graphics context with the given colo as foreground color */
  
  GC gc;                                /* handle of newly created GC.  */
  unsigned long valuemask = 0;          /* which values in 'values'  to */
                                        /* check when creating the GC.  */
                                        /* check when creating the GC.  */
  XGCValues values;                     /* initial values for the GC.   */
  unsigned int line_width = 2;          /* line width for the GC.       */
  int line_style = LineSolid;           /* style for lines drawing and  */
  int cap_style = CapButt;              /* style of the line's edje and */
  int join_style = JoinBevel;           /*  joined lines.               */
  int screen_num = DefaultScreen(display);

  Colormap colormap;
  XColor col;
  
  gc = XCreateGC(display, win, valuemask, &values);
  if (gc < 0) {
        fprintf(stderr, "XCreateGC: \n");
  }
  XSetSubwindowMode ( display, gc, IncludeInferiors );
  
  /* allocate foreground and background colors for this GC. */
  colormap = DefaultColormap (display, 0);
  XParseColor (display, colormap, color, &col);
  XAllocColor (display, colormap, &col);

  XSetForeground(display, gc, col.pixel);
  XSetBackground(display, gc, BlackPixel(display, screen_num));
  
  /* define the style of lines that will be drawn using this GC. */
  XSetLineAttributes(display, gc,
                     line_width, line_style, cap_style, join_style);
  
  /* define the fill style for the GC. to be 'solid filling'. */
  XSetFillStyle(display, gc, FillSolid);

  return gc;
}

int 
DrawText (Display* display, Window window, cfg_t cfg, 
           char *text)
{
  /* display the text on the window using a shadowy effect 
   * create a shaped window for it
   */

  Pixmap shapemask;
  GC gc_shape, gc_text, gc_border;
  XGCValues gcv;

  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  gcv.foreground = WhitePixel (display, 0);
  gcv.subwindow_mode = IncludeInferiors;

  shapemask = XCreatePixmap (display, window, MSG_W, MSG_H, 1);
  gc_shape = XCreateGC (display, shapemask, 0 , &gcv);
  gc_text = CreateGC (display, window, cfg.color_name);
  gc_border = CreateGC (display, window, "black");

  /* Do font stuff */

  /* assign the given font to our GC */
  XSetFont (display, gc_text, cfg.font_info->fid);
  XSetFont (display, gc_border, cfg.font_info->fid);
  XSetFont (display, gc_shape, cfg.font_info->fid);

  /* clear the mask */
  XSetForeground (display, gc_shape, 0);
  XFillRectangle (display, shapemask, gc_shape, 0, 0, MSG_W, MSG_H);
  XSetForeground (display, gc_shape, 1);
  
  {
    /* variables used for drawing the text strings. */
    int x, y;
    int x_off, y_off;
  

    x = 0;
    y = cfg.font_height;    

    /* draw black border part */
    for (x_off = 0; x_off <= 4; ++x_off)
    {
      for (y_off = 0; y_off <=4; ++y_off)
      {
        /* draw both border text and shape mask text */
        XDrawString (display, window, gc_border, x + x_off, y + y_off,
                     text,strlen(text));
        XDrawString (display, shapemask, gc_shape, x + x_off, y + y_off,
		     text, strlen (text));
      }
    }

    /* draw colored text */
    XDrawString (display, window, gc_text, x + 2, y + 2,
                     text,strlen(text));

    /* add shape mask to window */
    XShapeCombineMask (display, window, ShapeBounding, 0, 0, 
                       shapemask, ShapeSet);

    /* move the window to center the text */
    
    x = 0;
    y = (MSG_H - cfg.font_height) / 2 + cfg.line_number * MSG_H;

    return 0;
  }
}

int DrawXpm 
(Display* display, Window window, cfg_t cfg)
{
  Pixmap pixmap, shapemask;
  XpmAttributes attributes;
  XGCValues gcv;
  GC gc_clip;
  int x, y;
  int return_value;

  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  gcv.foreground = WhitePixel (display, 0);
  gcv.subwindow_mode = IncludeInferiors;

  attributes.valuemask = 0;		/* It is important to set the value mask ! */
  gc_clip = XCreateGC (display, window, GCFunction|GCGraphicsExposures, &gcv);

  DEBUG ("reading pixmap\n");
  return_value = XpmReadFileToPixmap (display, window, cfg.icon_file, 
  									  &pixmap, &shapemask, &attributes);

  if (return_value != 0)
  {
    printf ("ERROR: XpmReadFileToPixmap failed with value %d\n", return_value);
    return 1;
  }

  XShapeCombineMask (display, window, ShapeBounding, 0, 0, shapemask, ShapeSet);
  XCopyArea (display, pixmap, window, gc_clip, 0, 0, 100, 100, 0, 0);

  /* Calculate where to move the window if you want to center it */
  x = (IMG_W - attributes.width) / 2;
  y = (IMG_H - attributes.height) / 2 + cfg.line_number * MSG_H;
  XMoveWindow (display, window, x, y);

  /* clean up stuff */
     
  return 1;
}

int main (int argc, char *argv[])
{
  Display* display;             /* pointer to X Display structure.           */
  Window img_w, msg_w;
  char *display_name = getenv("DISPLAY");  /* address of the X display.      */
  int screen_num;               /* number of screen to place the window on.  */
  cfg_t cfg;					/* save our configuration options */
  int y;

  DEBUG ("DEBUG: parsing config options\n");

  if (!parse_config (argc, argv, &cfg))
  {
    usage ("Invalid configuration options");
    return 1;
  }

  if (argc == optind)
  {
    usage ("Please supply a line of text to display !");
    return 1;
  }

  /* open connection with the X server. */
  display = XOpenDisplay(display_name);
  if (display == NULL) 
  {
    fprintf(stderr, "%s: cannot connect to X server '%s'\n",
            argv[0], display_name);
    exit(1);
  }

  /* load the font */
  LoadFont (display, &cfg);
  
  /* get the geometry of the default screen for our display. */
  screen_num = DefaultScreen (display);
  cfg.display_width = DisplayWidth (display, screen_num);
  cfg.display_height = DisplayHeight (display, screen_num);

  /* calculate geometries for window */
  y = cfg.line_number * MSG_H;
  
  img_w = CreateWindow (display, screen_num,  0, y, IMG_W - 1, y + IMG_H - 1);
  msg_w = CreateWindow (display, screen_num, 50, y, 
                        cfg.display_width - IMG_W, y + MSG_H - 1);

  DrawText (display, msg_w, cfg, argv[argc - 1]);
  if (cfg.icon_file != NULL)
  {
    DrawXpm (display, img_w, cfg);
  }

  XFlush(display);
  usleep(cfg.duration * 1.0E6);

  /* close the connection to the X server. */
  XCloseDisplay(display);

  return 0;
}

void usage (const char *text)
{
  printf ("%s\n\n", text);
  printf ("Usage info :\n\n");
  printf ("Xmsg \t[-i (icon .xpm)] [-c (X color)] [-l (line number)]\n");
  printf ("     \t[-d (duration)] [-f (X font descr)] (text to display)\n");
}
